#include <stack>
#include <chrono>

#include "Quicksort.h"

using namespace std;
using namespace std::chrono;

/**
 * Public sort() function to sort the entire array.
 */
void Quicksort::sort()
{
    sorted_count = 0;

    subarray_stack.push(pair<int, int>(0, size - 1));
    sort_subarray();
}

/**
 * Private sort() function to sort subarrays
 * whose bounds are popped from the subarray stack.
 */
void Quicksort::sort_subarray()
{
    // Loop until the entire array is sorted.
    while (sorted_count < size)
    {
        // Pop off the bounds of a subarray as a pair object.
        pair<int, int> bounds_pair = subarray_stack.top();
        int left_index  = bounds_pair.first;
        int right_index = bounds_pair.second;
        subarray_stack.pop();

        // Process the subarray.
        process_subarray(left_index, right_index);
    }
}

/**
 * Process a subarray whose bounds were popped off the stack.
 * Keep track of the count of sorted elements.
 * @param left_index the leftmost index of the subarray.
 * @param right_index the rightmost index of the subarray.
 */
void Quicksort::process_subarray(const int left_index,
                                 const int right_index)
{
    int subarray_size = right_index - left_index + 1;

    // Base cases: Subarray sizes of 0, 1, and 2.
    if      (subarray_size <= 0) return;
    else if (subarray_size == 1) ++sorted_count;
    else if (subarray_size == 2)
    {
        // Swap them if necessary.
        if (data[left_index] > data[right_index])
        {
            std::swap(data[left_index], data[right_index]);
        }

        sorted_count += 2;
    }

    // Subarray size > 2: Partition this subarray and push the
    // bounds of the resulting two smaller subarrays onto the stack.
    else
    {
        int pivot_index = partition(left_index, right_index);
        ++sorted_count;

        // New bounds.
        int next_right = pivot_index - 1;
        int next_left  = pivot_index + 1;

        // Push the two smaller subarrays onto the stack.
        if (next_right >= 0)  push_subarray(left_index, next_right);
        if (next_left < size) push_subarray(next_left,  right_index);
    }
}

/**
 * Push the bounds of a subarray onto the subarray stack.
 * @param left_index the leftmost index of the subarray.
 * @param right_index the rightmost index of the subarray.
 */
void Quicksort::push_subarray(const int left_index,
                              const int right_index)
{
    pair<int, int> bounds_pair(left_index, right_index);
    subarray_stack.push(bounds_pair);
}

/**
 * Partition a subarray bounded by its leftmost and rightmost indexes.
 * @param left_index the leftmost index of the subarray.
 * @param right_index the rightmost index of the subarray.
 * @return the index of the pivot element.
 */
int Quicksort::partition(const int left_index,
                         const int right_index)
{
    int middle_index = (left_index + right_index)/2;

    int *mid   = &data[middle_index];
    int *left  = &data[left_index];
    int *right = &data[right_index];

    int pivot_value = data[middle_index];
    std::swap(*mid, *right);

    int *i = left - 1;  // point i to the left end
    int *j = right;     // point j tlo the right end

    do
    {
        while (*(++i) < pivot_value) {}  // move pointer i to the left
        while (*(--j) > pivot_value) {}  // move pointer j to the right

        // Swap the ith and jth elements after i and j stop.
        if (i < j) std::swap(*i, *j);
    } while (i < j);  // break out when i and j cross

    // Swap the pivot element back in.
    std::swap(*i, *right);

    // Return the index of the pivot element.
    return i - &data[0];
}

/**
 * Paranoid programming: Verify that the array was successfully sorted.
 * @return true if sorted, false if not.
 */
bool Quicksort::verify_sorted() const
{
    for (int i = 1; i < size; i++)
    {
        if (data[i] < data[i-1]) return false;
    }

    return true;
}
