#include <climits>
#include <vector>
#include <climits>
#include <algorithm>

#include "Island.h"
#include "Bridge.h"
#include "Path.h"
#include "PathFinder.h"

using namespace std;

/**
 * Recursive function that returns a vector of the cheapest
 * paths from the current island (where we are now) to the
 * destination island.
 * @param current the current island.
 * @return a vector of the cheapest paths.
 */
vector<Path *> *PathFinder::find_cheapest_paths(const Island *current)
{
    // A vector of all the good paths, the ones that lead
    // from the current island to the destination island. We
    // will select the cheapest paths from among these paths.
    vector<Path *> good_paths;

    // BASE CASE #1: The current island is the destination island.
    //               Create a path containing only the destination
    //               island and add it to goodPaths. There won't
    //               be a recursive call.
    if (current == get_destination())
    {
        Path *path = new Path();

        path->add_island(get_destination());
        good_paths.push_back(path);
    }

    // Recursive case: Find the cheapest paths leading from the
    //                 current island to the destination island.
    else
    {
        // Examine each of the current island's outgoing bridges.
        //
        // BASE CASE #2: An island without any outgoing bridges.
        //               The for loop below won't execute and
        //               there won't be any recursive calls.
        //
        // "Crossing a bridge" to its endpoint island brings us
        // one step closer to a base case. The for loop below
        // recursively finds the cheapest paths (if any) from each
        // bridge's endpoint island to the destination island.
        //
        // BACKTRACK: Do the same for the island's next bridge.
        //            After trying all the bridges at the current
        //            island, return to continue trying the
        //            bridges at the previous island.
        //
        for (const Bridge * const bridge : current->get_bridges())
        {
            // Recursive call: Call on the endpoint island of the
            //                 bridge to return a vector of the
            //                 cheapest paths starting from that island.
            const Island *bridge_endpoint = bridge->get_endpoint();
            vector<Path *> *endpoint_paths =
                                    find_cheapest_paths(bridge_endpoint);

            // Did the endpoint island have any outgoing bridges?
            // If not, that was base case #2. No recursive call
            // and the endpoint island is not appended to the path.
            if (!endpoint_paths->empty())
            {
                // Append the current island to each endpoint path
                // and increase the path's cost by the bridge toll.
                // (A path will contain the islands on its route
                // in reverse order.)
                for (Path *path : *endpoint_paths)
                {
                    path->add_island(current);
                    path->increase_cost(bridge->get_toll());
                }

                // Collect all the current island's good paths.
                good_paths.insert(good_paths.end(),
                                  endpoint_paths->begin(),
                                  endpoint_paths->end());
            }
        }
    }

    vector<Path *> *cheapest_paths = new vector<Path *>();

    // What's the lowest cost among
    // the current island's good paths?
    int lowest_cost = INT_MAX;
    for (Path *path : good_paths)
    {
        lowest_cost = std::min(lowest_cost, path->get_cost());
    }

    // Select only the cheapest paths among
    // the current island's good paths.
    for (Path *path : good_paths)
    {
        if (path->get_cost() == lowest_cost)
        {
            cheapest_paths->push_back(path);
        }
        else
        {
            delete path;
        }
    }

    return cheapest_paths;
}
